/* ***********************************************************
 * Smart Cards: Toward a Modern Run-time Platform
 * ETH Zurich, WS 2005/2006
 *************************************************************/

package perftestclient;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.text.DecimalFormat;

import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.ibm.jc.JCTerminal;
import com.ibm.jc.JCard;
import com.ibm.jc.terminal.TimeJCTerminal;
import com.ibm.jc.terminal.TraceJCTerminal;

public class PerfTestClient {

	public static void main(String[] args) {
		new PerfTestClient().run();
	}

	private PrintWriter traceWriter;

	private void run() {
		Display display = new Display();
		Shell shell = new Shell(display);

		shell.setLayout(new GridLayout());
		shell.setText("Performance Optimization Demonstration");
		createControls(shell).setLayoutData(new GridData(GridData.FILL_BOTH));

		shell.open();

		while (!shell.isDisposed()) {
			try {
				if (!display.readAndDispatch())
					display.sleep();
			} catch (Exception e) {
				System.err.println("Uncaught exception in event loop:");
				e.printStackTrace();
				System.err.println();
			}
		}
		display.dispose();
	}

	private JCTerminal injectTracer(JCTerminal term) {
		TraceJCTerminal tt = new TraceJCTerminal();
		tt.setLog(traceWriter);
		tt.filterFor(term);
		return tt;
	}

	private TimeJCTerminal tt;

	private PerfTestApplet applet;

	private DecimalFormat df = new DecimalFormat("0.00");

	private Control createControls(final Shell parent) {
		Composite co = new Composite(parent, SWT.NONE);
		co.setLayout(new GridLayout(2, true));

		GridData gd;

		final Combo cmbTerminal = new Combo(co, SWT.NONE);
		gd = new GridData();
		gd.horizontalSpan = 2;
		gd.widthHint = 300;
		cmbTerminal.setLayoutData(gd);
		cmbTerminal.add("winscard:4");
		cmbTerminal.add("Remote");
		cmbTerminal.select(0);

		final Button butConnect = new Button(co, SWT.PUSH);
		butConnect.setText("Connect");
		gd = new GridData();
		gd.horizontalSpan = 3;
		butConnect.setLayoutData(gd);

		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gd.horizontalSpan = 2;
		new Label(co, SWT.SEPARATOR | SWT.HORIZONTAL).setLayoutData(gd);

		final Combo cmbTests = new Combo(co, SWT.READ_ONLY);
		gd = new GridData();
		gd.horizontalSpan = 2;
		gd.widthHint = 300;
		cmbTests.setLayoutData(gd);

		final Button butRun = new Button(co, SWT.PUSH);
		butRun.setText("Run Test");
		gd = new GridData();
		gd.horizontalSpan = 2;
		butRun.setLayoutData(gd);

		Label lblSTitle = new Label(co, SWT.NONE | SWT.CENTER);
		lblSTitle.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		lblSTitle.setFont(JFaceResources.getBannerFont());
		lblSTitle.setText("Basic Version");

		Label lblFTitle = new Label(co, SWT.NONE | SWT.CENTER);
		lblFTitle.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		lblFTitle.setFont(JFaceResources.getBannerFont());
		lblFTitle.setText("Optimized Version");

		final Label lblSlow = new Label(co, SWT.BORDER);
		lblSlow.setFont(JFaceResources.getHeaderFont());
		lblSlow.setText(" \n \n ");
		lblSlow.setAlignment(SWT.CENTER);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		lblSlow.setLayoutData(gd);

		final Label lblFast = new Label(co, SWT.BORDER);
		lblFast.setFont(JFaceResources.getHeaderFont());
		lblFast.setText(" \n \n ");
		lblFast.setAlignment(SWT.CENTER);
		gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		lblFast.setLayoutData(gd);

		final Text txtTrace = new Text(co, SWT.MULTI | SWT.V_SCROLL
				| SWT.H_SCROLL | SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 2;
		txtTrace.setLayoutData(gd);
		txtTrace.setFont(JFaceResources.getTextFont());

		traceWriter = new PrintWriter(new Writer() {
			private boolean closed = false;

			private char[] intbuf = new char[1024];

			private int intbuf_off = 0;

			private int intbuf_left = intbuf.length;

			public void write(char[] b, int off, int len) throws IOException {
				if (closed) {
					throw new IOException();
				}
				while (len > 0) {
					int n = Math.min(intbuf_left, len);
					int_write(b, off, n);
					len -= n;
					off += n;
				}
			}

			private void int_write(char[] cbuf, int off, int len) {
				System.arraycopy(cbuf, off, intbuf, intbuf_off, len);
				intbuf_left -= len;
				if (intbuf_left > 0) {
					intbuf_off += len;
				} else {
					write_out();
				}
			}

			private void write_out() {
				final String chunk = new String(intbuf, 0, intbuf_off);
				intbuf_left = intbuf.length;
				intbuf_off = 0;
				txtTrace.getDisplay().asyncExec(new Runnable() {
					public void run() {
						txtTrace.append(chunk);
					}
				});
			}

			public void close() throws IOException {
				if (closed) {
					return;
				}
				flush();
				closed = true;
			}

			public void flush() throws IOException {
				if (closed) {
					throw new IOException();
				}
				write_out();
			}
		});

		butConnect.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				JCTerminal term = JCTerminal.getInstance(cmbTerminal.getText(),
						null);
				term.open();
				term = injectTracer(term);

				tt = new TimeJCTerminal();
				tt.init(term);

				applet = new PerfTestApplet(new JCard(tt, null, 500));
				applet.select();

				cmbTests.removeAll();
				int n = applet.getNoTests();
				if (n > 0) {
					for (int i = 0; i < n; i++) {
						cmbTests.add(applet.getTestName(i));
					}
					cmbTests.select(0);
				}
			}
		});

		butRun.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				lblSlow.setText(" \n \n ");
				lblFast.setText(" \n \n ");

				int no = cmbTests.getSelectionIndex();
				tt.reset();
				applet.runOverHead(no);
				long overhead = tt.getCmdTime(0);
				tt.reset();
				applet.runTest(no);
				long t1 = tt.getCmdTime(0) - overhead;
				tt.reset();
				applet.runOptimizedTest(no);
				long t2 = tt.getCmdTime(0) - overhead;

				lblSlow.setText(" \n" + t1 + "\n ");
				lblFast
						.setText(" \n" + t2 + " ("
								+ df.format((double) t1 / (double) t2)
								+ "x faster)\n ");
			}
		});

		return co;
	}
}