/* ***********************************************************
 * Smart Cards: Toward a Modern Run-time Platform
 * ETH Zurich, WS 2005/2006
 *************************************************************/

package purse.offcard;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.rmi.RemoteException;

import javacard.framework.UserException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import purse.PurseInterface;

import com.ibm.jc.JCApplet;
import com.ibm.jc.JCTerminal;
import com.ibm.jc.JCard;
import com.ibm.jc.rmi.RMIObjectFactory;
import com.ibm.jc.terminal.TraceJCTerminal;

public class OffcardPurse {

	private static class InvalidAmountException extends Exception {

	}

	private static class MaxBalanceExceededException extends Exception {

	}

	private static class NegativeBalanceException extends Exception {

	}

	public static void main(String[] args) {
		new OffcardPurse().run();
	}

	private PrintWriter traceWriter;

	private void run() {
		Display display = new Display();
		Shell shell = new Shell(display);

		shell.setLayout(new GridLayout());
		createControls(shell).setLayoutData(new GridData(GridData.FILL_BOTH));

		shell.open();

		while (!shell.isDisposed()) {
			try {
				if (!display.readAndDispatch())
					display.sleep();
			} catch (Exception e) {
				System.err.println("Uncaught exception in event loop:");
				e.printStackTrace();
				System.err.println();
			}
		}
		display.dispose();
	}

	private JCTerminal injectTracer(JCTerminal term) {
		TraceJCTerminal tt = new TraceJCTerminal();
		tt.setLog(traceWriter);
		tt.filterFor(term);
		return tt;
	}

	private Control createControls(final Shell parent) {
		Composite co = new Composite(parent, SWT.NONE);
		co.setLayout(new GridLayout(3, false));

		GridData gd;

		final Combo cmbTerminal = new Combo(co, SWT.NONE);
		gd = new GridData();
		gd.horizontalSpan = 3;
		cmbTerminal.setLayoutData(gd);
		cmbTerminal.add("Remote");
		cmbTerminal.add("winscard:4");
		cmbTerminal.select(0);

		final Button butConnect = new Button(co, SWT.PUSH);
		butConnect.setText("Connect");
		gd = new GridData();
		gd.horizontalSpan = 3;
		butConnect.setLayoutData(gd);

		final Button butGetBalance = new Button(co, SWT.PUSH);
		butGetBalance.setText("Get Balance");
		gd = new GridData();
		gd.horizontalSpan = 3;
		butGetBalance.setLayoutData(gd);

		final Text txtAmount = new Text(co, SWT.SINGLE | SWT.BORDER);

		final Button butCredit = new Button(co, SWT.PUSH);
		butCredit.setText("Credit");
		gd = new GridData();
		butCredit.setLayoutData(gd);

		final Button butDebit = new Button(co, SWT.PUSH);
		butDebit.setText("Debit");
		gd = new GridData();
		butDebit.setLayoutData(gd);

		final Text txtTrace = new Text(co, SWT.MULTI | SWT.V_SCROLL
				| SWT.H_SCROLL | SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		gd.horizontalSpan = 3;
		txtTrace.setLayoutData(gd);
		txtTrace.setFont(JFaceResources.getTextFont());

		traceWriter = new PrintWriter(new Writer() {
			private boolean closed = false;

			private char[] intbuf = new char[1024];

			private int intbuf_off = 0;

			private int intbuf_left = intbuf.length;

			public void write(char[] b, int off, int len) throws IOException {
				if (closed) {
					throw new IOException();
				}
				while (len > 0) {
					int n = Math.min(intbuf_left, len);
					int_write(b, off, n);
					len -= n;
					off += n;
				}
			}

			private void int_write(char[] cbuf, int off, int len) {
				System.arraycopy(cbuf, off, intbuf, intbuf_off, len);
				intbuf_left -= len;
				if (intbuf_left > 0) {
					intbuf_off += len;
				} else {
					write_out();
				}
			}

			private void write_out() {
				final String chunk = new String(intbuf, 0, intbuf_off);
				intbuf_left = intbuf.length;
				intbuf_off = 0;
				txtTrace.getDisplay().asyncExec(new Runnable() {
					public void run() {
						txtTrace.append(chunk);
					}
				});
			}

			public void close() throws IOException {
				if (closed) {
					return;
				}
				flush();
				closed = true;
			}

			public void flush() throws IOException {
				if (closed) {
					throw new IOException();
				}
				write_out();
			}
		});

		butConnect.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				connect(cmbTerminal.getText(), null);
			}
		});

		butGetBalance.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent ev) {
				short balance = getBalance();
				MessageDialog.openInformation(parent, null,
						"The current balance is: " + balance);
			}
		});

		butCredit.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent ev) {
				try {
					credit(Short.parseShort(txtAmount.getText()));
				} catch (InvalidAmountException e) {
					MessageDialog.openError(parent, null, "Invalid amount!");
				} catch (MaxBalanceExceededException e) {
					MessageDialog.openError(parent, null,
							"Maximum balance exceeded!");
				}
			}
		});

		butDebit.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent ev) {
				try {
					debit(Short.parseShort(txtAmount.getText()));
				} catch (InvalidAmountException e) {
					MessageDialog.openError(parent, null, "Invalid amount!");
				} catch (NegativeBalanceException e) {
					MessageDialog.openError(parent, null, "Negative balance!");
				}
			}
		});

		return co;
	}

	private static final byte[] APP_AID = "purse.app".getBytes();

	private PurseInterface purse;

	private void connect(String termSpec, String optSpec) {
		// TODO: connect
		JCTerminal term = JCTerminal.getInstance(termSpec, optSpec);
		term.open();

		term = injectTracer(term);

		JCard card = new JCard(term, null, 500);
		JCApplet jcapp = new JCApplet(card, APP_AID);

		purse = (PurseInterface) RMIObjectFactory.getInitialReference(jcapp);
	}

	private short getBalance() {
		// TODO: get balance
		try {
			return purse.getBalance();
		} catch (RemoteException e) {
			throw new RuntimeException(e);
		}
	}

	private void credit(short amount) throws InvalidAmountException,
			MaxBalanceExceededException {
		// TODO: credit
		try {
			purse.credit(amount);
		} catch (UserException e) {
			if (e.getReason() == PurseInterface.SW_INVALID_AMOUNT) {
				throw new InvalidAmountException();
			} else {
				throw new MaxBalanceExceededException();
			}
		} catch (RemoteException e) {
			throw new RuntimeException(e);
		}
	}

	private void debit(short amount) throws InvalidAmountException,
			NegativeBalanceException {
		try {
			purse.debit(amount);
		} catch (UserException e) {
			if (e.getReason() == PurseInterface.SW_INVALID_AMOUNT) {
				throw new InvalidAmountException();
			} else {
				throw new NegativeBalanceException();
			}
		} catch (RemoteException e) {
			throw new RuntimeException(e);
		}
	}
}