/* ***********************************************************
 * Smart Cards: Toward a Modern Run-time Platform
 * ETH Zurich, WS 2005/2006
 * 
 * Exercise 4 - Solution
 *************************************************************/

package vault;

import java.rmi.RemoteException;

import javacard.framework.APDU;
import javacard.framework.Applet;
import javacard.framework.ISO7816;
import javacard.framework.ISOException;
import javacard.framework.JCSystem;
import javacard.framework.OwnerPIN;
import javacard.framework.SystemException;
import javacard.framework.UserException;
import javacard.framework.Util;
import javacard.framework.service.CardRemoteObject;
import javacard.framework.service.Dispatcher;
import javacard.framework.service.RMIService;

public class VaultApplet extends Applet implements Vault {

	private static final byte MAX_PIN_TRIES = 3;

	private static final byte MIN_PIN_LENGTH = 4;

	private static final byte MAX_PIN_LENGTH = 8;

	private static final byte MAX_PUK_TRIES = 10;

	private static final byte MIN_PUK_LENGTH = 8;

	private static final byte MAX_PUK_LENGTH = 8;

	private static final byte INS_RMI = 0x70;

	private final OwnerPIN PIN;

	private final OwnerPIN PUK;

	private PasswordEntry root;

	private final Dispatcher dispatcher;

	public static void install(byte[] bArray, short bOffset, byte bLength) {
		new VaultApplet(bArray, bOffset, bLength).register(bArray,
				(short) (bOffset + 1), bArray[bOffset]);
	}

	private VaultApplet(byte[] bArray, short bOffset, byte bLength) {
		// skip instance AID
		bOffset += (short) (bArray[bOffset] + 1);
		// skip application privileges
		bOffset += (short) (bArray[bOffset] + 1);

		// Now we can look at the install parameters
		short n = bArray[bOffset];
		if (n == 0) {
			ISOException.throwIt(ISO7816.SW_WRONG_DATA);
		}
		byte pin_len = bArray[bOffset + 1];
		if (pin_len < MIN_PIN_LENGTH || n < pin_len + 2) {
			ISOException.throwIt(ISO7816.SW_WRONG_DATA);
		}
		byte puk_len = bArray[bOffset + pin_len + 2];
		if (puk_len < MIN_PUK_LENGTH || n < pin_len + 2 + puk_len) {
			ISOException.throwIt(ISO7816.SW_WRONG_DATA);
		}
		PIN = new OwnerPIN(MAX_PIN_TRIES, MAX_PIN_LENGTH);
		PIN.update(bArray, (short) (bOffset + 2), pin_len);
		PUK = new OwnerPIN(MAX_PUK_TRIES, MAX_PUK_LENGTH);
		PUK.update(bArray, (short) (bOffset + 3 + pin_len), puk_len);

		CardRemoteObject.export(this);
		RMIService service = new RMIService(this);
		service.setInvokeInstructionByte(INS_RMI);
		dispatcher = new Dispatcher((short) 1);
		dispatcher.addService(service, Dispatcher.PROCESS_COMMAND);
	}

	public void process(APDU apdu) {
		// Good practice: Return 9000 on SELECT
		if (selectingApplet()) {
			dispatcher.process(apdu);
			return;
		}

		byte[] buf = apdu.getBuffer();
		byte n = (byte) apdu.setIncomingAndReceive();

		try {
			switch (buf[ISO7816.OFFSET_INS]) {
			case (byte) 0x00: // verify PIN
				verifyPIN(buf, ISO7816.OFFSET_CDATA, n);
				break;
			case (byte) 0x02: // update PIN
				updatePIN(buf, ISO7816.OFFSET_CDATA, n);
				break;
			case (byte) 0x04: // unblock PIN
				unblockPIN(buf, ISO7816.OFFSET_CDATA, n);
				break;
			case (byte) 0x10: // get pass
				byte[] pass = getPassword(buf, ISO7816.OFFSET_CDATA, n);
				Util.arrayCopy(pass, (short) 0, buf, (short) 0,
						(n = (byte) pass.length));
				apdu.setOutgoingAndSend((short) 0, n);
				break;
			case (byte) 0x12: // save pass
				n = buf[ISO7816.OFFSET_CDATA];
				savePassword(buf, (short) (ISO7816.OFFSET_CDATA + 1), n, buf,
						(short) ((n += (ISO7816.OFFSET_CDATA + 1)) + 1), buf[n]);
				break;
			case (byte) 0x14: // remove pass
				removePassword(buf, ISO7816.OFFSET_CDATA, n);
				break;
			case INS_RMI: // RMI
				dispatcher.process(apdu);
				break;
			default: // huh?
				ISOException.throwIt(ISO7816.SW_INS_NOT_SUPPORTED);
			}
		} catch (UserException e) {
			// convert user exceptions to ISO
			ISOException.throwIt(e.getReason());
		}
	}

	public void verifyPIN(byte[] pin, short off, byte len) throws UserException {
		if (PIN.getTriesRemaining() == 0) {
			UserException.throwIt(SW_PIN_BLOCKED);
		}
		if (!PIN.check(pin, off, len)) {
			UserException.throwIt(SW_WRONG_PIN);
		}
	}

	public void updatePIN(byte[] pin, short off, byte len) throws UserException {
		if (!PIN.isValidated()) {
			UserException.throwIt(SW_NOT_AUTHORIZED);
		}
		PIN.update(pin, off, len);
	}

	public void unblockPIN(byte[] puk, short off, byte len)
			throws UserException {
		if (PIN.getTriesRemaining() > 0) {
			UserException.throwIt(SW_PIN_NOT_BLOCKED);
		}
		if (!PUK.check(puk, off, len)) {
			UserException.throwIt(SW_WRONG_PIN);
		}
		PIN.resetAndUnblock();
	}

	public byte[] getPassword(byte[] URL, short off, byte len)
			throws UserException {
		if (!PIN.isValidated()) {
			UserException.throwIt(SW_NOT_AUTHORIZED);
		}
		PasswordEntry entry = root;
		while (entry != null) {
			if (entry.match(URL, off, len)) {
				return entry.getPass();
			}
			entry = entry.next;
		}
		UserException.throwIt(SW_NOT_FOUND);
		return null; // not reached;
	}

	public void savePassword(byte[] URL, short uoff, byte ulen, byte[] pass,
			short poff, byte plen) throws UserException {
		if (!PIN.isValidated()) {
			UserException.throwIt(SW_NOT_AUTHORIZED);
		}
		PasswordEntry last = null;
		PasswordEntry entry = root;
		while (entry != null) {
			if (entry.match(URL, uoff, ulen)) {
				UserException.throwIt(SW_ALREADY_EXISTS);
			}
			last = entry;
			entry = entry.next;
		}
		try {
			if (last == null) {
				root = new PasswordEntry(URL, uoff, ulen, pass, poff, plen);
			} else {
				last.next = new PasswordEntry(URL, uoff, ulen, pass, poff, plen);
			}
		} catch (SystemException e) {
			if (e.getReason() == SystemException.NO_RESOURCE) {
				UserException.throwIt(SW_MEMORY_FULL);
			} else {
				throw e;
			}
		}
	}

	public void removePassword(byte[] URL, short off, byte len)
			throws UserException {
		if (!PIN.isValidated()) {
			UserException.throwIt(SW_NOT_AUTHORIZED);
		}
		PasswordEntry last = null;
		PasswordEntry entry = root;
		while (entry != null) {
			if (entry.match(URL, off, len)) {
				if (last != null) {
					last.next = entry.next;
				} else {
					root = entry.next;
				}
				if (JCSystem.isObjectDeletionSupported()) {
					JCSystem.requestObjectDeletion();
				}
				return;
			}
			last = entry;
			entry = entry.next;
		}
		UserException.throwIt(SW_NOT_FOUND);
	}

	public void verifyPIN(byte[] pin) throws UserException, RemoteException {
		verifyPIN(pin, (short) 0, (byte) pin.length);
	}

	public void updatePIN(byte[] pin) throws UserException, RemoteException {
		updatePIN(pin, (short) 0, (byte) pin.length);
	}

	public void unblockPIN(byte[] puk) throws UserException, RemoteException {
		unblockPIN(puk, (short) 0, (byte) puk.length);
	}

	public byte[] getPassword(byte[] URL) throws UserException, RemoteException {
		return getPassword(URL, (short) 0, (byte) URL.length);
	}

	public void savePassword(byte[] URL, byte[] pass) throws UserException,
			RemoteException {
		savePassword(URL, (short) 0, (byte) URL.length, pass, (short) 0,
				(byte) pass.length);
	}

	public void removePassword(byte[] URL) throws UserException,
			RemoteException {
		removePassword(URL, (short) 0, (byte) URL.length);
	}
}
